// src/controllers/reportController.js
const { Report, Observation, User } = require('../models');

// 1. إنشاء تقرير جديد
exports.createReport = async (req, res) => {
    try {
        // استخراج البيانات (مع مراعاة الصور والملفات)
        const facilityName = req.body.facilityName;
        const location = req.body.location;
        const reportNumber = req.body.reportNumber;
        const visitDate = req.body.visitDate;
        
        // تحويل النصوص JSON إلى كائنات
        // نستخدم try/catch صغير هنا لأن البيانات قد تأتي كـ object جاهز أو string حسب طريقة الإرسال
        let checklistData, observationsData;
        try { checklistData = typeof req.body.checklistData === 'string' ? JSON.parse(req.body.checklistData) : req.body.checklistData; } catch(e) { checklistData = {}; }
        try { observationsData = typeof req.body.observations === 'string' ? JSON.parse(req.body.observations) : req.body.observations; } catch(e) { observationsData = []; }

        // إنشاء التقرير
        const newReport = await Report.create({
            reportNumber,
            facilityName,
            location,
            visitDate,
            checklistData,
            inspectorId: req.userData.userId,
            status: 'pending_supervisor'
        });

        // إنشاء الملاحظات وربط الصور
        if (observationsData && observationsData.length > 0) {
            const observationsToSave = observationsData.map((obs, index) => {
                const imageFieldName = `obs_image_${index}`;
                // البحث في الملفات المرفوعة (إن وجدت)
                const uploadedFile = req.files ? req.files.find(f => f.fieldname === imageFieldName) : null;

                return {
                    reportId: newReport.id,
                    description: obs.description,
                    requiredAction: obs.requiredAction,
                    severity: obs.severity,
                    correctionDuration: obs.correctionDuration,
                    status: 'مفتوحة',
                    violationImage: uploadedFile ? `/uploads/${uploadedFile.filename}` : null
                };
            });
            await Observation.bulkCreate(observationsToSave);
        }

        res.status(201).json({ message: "تم حفظ التقرير بنجاح", reportId: newReport.id });
    } catch (error) {
        console.error("Error creating report:", error);
        res.status(500).json({ message: "حدث خطأ أثناء المعالجة: " + error.message });
    }
};

// 2. جلب جميع التقارير
exports.getAllReports = async (req, res) => {
    try {
        const reports = await Report.findAll({
            include: [
                { model: User, as: 'inspector', attributes: ['fullName'] }
            ],
            order: [['createdAt', 'DESC']]
        });
        res.json(reports);
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
};

// 3. جلب تفاصيل تقرير واحد (هذه هي الدالة التي كانت مفقودة أو تسبب المشكلة)
exports.getReportDetails = async (req, res) => {
    try {
        const report = await Report.findByPk(req.params.id, {
            include: [
                { model: User, as: 'inspector', attributes: ['fullName'] },
                { model: Observation, as: 'observations' }
            ]
        });
        if (!report) return res.status(404).json({ message: "التقرير غير موجود" });
        res.json(report);
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
};